/*
 *	JSCViews collection 1
 *	(SwingOSC classes for SuperCollider)
 *
 *	Copyright (c) 2005-2008 Hanns Holger Rutz. All rights reserved.
 *
 *	This software is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either
 *	version 2, june 1991 of the License, or (at your option) any later version.
 *
 *	This software is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License (gpl.txt) along with this software; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 *	For further information, please contact Hanns Holger Rutz at
 *	contact@sciss.de
 *
 *
 *	Changelog:
 */

/**
 *	@version		0.59, 30-Jan-07
 *	@author		Hanns Holger Rutz
 */
JSCContainerView : JSCView { // abstract class
	var <children, <>decorator;
	var pendingValidation = false;
			
	// ----------------- public instance methods -----------------

	removeAll {
		children.copy.do({ arg child; child.remove });
	}
	
	// ----------------- quasi-interface methods : crucial-lib support -----------------

	asPageLayout { arg title, bounds;
		// though it won't go multi page
		// FlowView better ?
		^MultiPageLayout.on( this, bounds );
	}

	flow { arg func, bounds;
		var f, comp;
		f = FlowView( this, bounds /*?? { this.bounds }*/ );
		func.value( f );
		f.resizeToFit;
		^f;
	}
	
	horz { arg func, bounds;
		var comp;
		comp = JSCHLayoutView.new( this, bounds ?? { this.bounds });
		func.value( comp );
		^comp;
	}
	
	vert { arg func, bounds;
		var comp;
		comp = JSCVLayoutView.new( this, bounds ?? { this.bounds });
		func.value( comp );
		^comp;
	}
	
	comp { arg func, bounds;
		var comp;
		comp = JSCCompositeView.new( this, bounds ?? { this.bounds });
		func.value( comp );
		^comp;
	}		

	// ----------------- private instance methods -----------------

	prViewPortID { ^id }	// actually this refers to a viewport-view!! we should fuse that with the containerID concept XXX
	prChildOrder { ^-1 }

	prGetRefTopLeft {
// more efficient but too difficult to maintain
//		var refTopLeft;
//		var pTopLeft;
//		if( scBounds.isNil, {
//			// need to revalidate bounds
//			pTopLeft		= parent.prGetRefTopLeft;
//			refTopLeft	= jBounds.moveBy( pTopLeft.x, pTopLeft.y );
//			scBounds		= jinsets.addTo( refTopLeft );
//			^refTopLeft;
//		}, {
//			^(scBounds.leftTop - jinsets.leftTop);
//		});
		^(this.bounds.leftTop - jinsets.leftTop);
	}

	add { arg child;
		var bndl, vpID;
		
		children = children.add( child );
		if( decorator.notNil, { decorator.place( child )});

		if( child.id.notNil, {
			vpID = this.prViewPortID;
			bndl = Array( 4 );
			bndl.add([ '/method', vpID, \add, '[', '/ref', child.prContainerID, ']', this.prChildOrder ]);
			if( this.prAllVisible, {
				if( this.id != vpID, {
					bndl.add([ '/method', vpID, \validate ]);
				});
				bndl.add([ '/method', this.id, \revalidate ]);
				bndl.add([ '/method', child.id, \repaint ]);
				pendingValidation = false;
			}, {
				pendingValidation = true;
			});
			server.listSendBundle( nil, bndl );
		});
	}
	
	prInvalidateBounds {
		scBounds = nil;
		children.do({ arg child;
//			child.prSetScBounds( nil );
			child.prInvalidateBounds;
		});
	}

	prInvalidateAllVisible {
		allVisible = nil;
		children.do({ arg child;
			child.prInvalidateAllVisible;
		});
	}

	prVisibilityChange { arg pre, post;
		var vpID;
		if( pendingValidation, {
			if( this.prAllVisible, {
				vpID = this.prViewPortID;
				if( this.id != vpID, {
					post.add([ '/method', vpID, \validate ]);
				});
				post.add([ '/method', this.id, \revalidate ]);
				post.add([ '/method', this.id, \repaint ]);
				pendingValidation = false;
			});
		});
		children.do({ arg child;
			child.prVisibilityChange( pre, post );
		});
	}

	prRemoveChild { arg child;
		var bndl, vpID;
		
		children.remove( child );
		bndl = Array( 4 );
		vpID = this.prViewPortID;
		bndl.add([ '/method', vpID, \remove, '[', '/ref', child.prContainerID, ']' ]);
		if( this.prAllVisible, {
			if( this.id != vpID, {
				bndl.add([ '/method', vpID, \validate ]);
			});
			bndl.add([ '/method', this.id, \revalidate ]);
			bndl.add([ '/method', this.id, \repaint ]);
			pendingValidation = false;
		}, {
			pendingValidation = true;
		});
		server.listSendBundle( nil, bndl );
		// ... decorator replace all
	}
	//bounds_  ... replace all

	prMoveChild { arg bndl, child;
		var vpID;
		if( child.prAllVisible, {
			vpID = this.prViewPortID;
			if( this.id != vpID, {
				bndl.add([ '/method', vpID, \validate ]);
			});
			bndl.add([ '/method', this.id, \revalidate ]);
			bndl.add([ '/method', child.id, \repaint ]);
			pendingValidation = false;
		}, {
			pendingValidation = true;
		});
	}

	prVisibleChild {}

	prClose { arg preMsg, postMsg;
		super.prClose( preMsg, postMsg );
		children.do({ arg item; item.prClose });
	}

	prSCViewNew { arg preMsg, postMsg;
		properties.put( \canFocus, false );
		^super.prSCViewNew( preMsg, postMsg );
	}
	
	protDraw {
		children.do({ arg child; child.protDraw });
	}

	prSendProperty { arg key, value;
		switch( key,
		\background, {	// overriden to redirect to viewport
			server.listSendMsg([ '/set', this.prViewPortID, key ] ++ value.asSwingArg );
			^nil;
		});
		^super.prSendProperty( key, value );
	}
}

JSCCompositeView : JSCContainerView {
	// ----------------- quasi-interface methods : crucial-lib support -----------------

	asFlowView { arg bounds;
		^FlowView( this,bounds ?? { this.bounds });
	}

	// ----------------- private instance methods -----------------

	prChildOrder { ^0 }

	prSCViewNew {
		jinsets = Insets( 3, 3, 3, 3 );  // so focus borders of children are not clipped
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.Panel", '[', '/new', "de.sciss.swingosc.ColliderLayout", ']', ']' ]
		]);
	}
}

JSCTopView : JSCContainerView {	// NOT subclass of JSCCompositeView
	var window;

	// ----------------- public instance methods -----------------

	focus { arg flag = true;
		if( flag, {
			server.sendMsg( '/method', this.id, \requestFocus );
		}, {
			"JSCTopView.focus( false ) : not yet implemented".error;
		});
	}

	findWindow { ^this.prGetWindow }
	
	// only in construction mode, handled internally
	canReceiveDrag { ^currentDrag.isKindOf( Class )}

//	// bug: visible_( false ) doesn't properly refresh the view ...
//	visible_ { arg bool;
//		var bndl; //, cntID, tempID;
//		if( visible != bool, {
//			visible = bool;	// must be set before calling prVisiblityChange
//			this.prInvalidateAllVisible;
//			bndl = List.new;
//			this.prVisibilityChange( bndl );
////			cntID = if( this.prIsInsideContainer, { "cn" ++ this.id }, { this.id });
//			bndl.add([ '/set', this.id, \visible, visible ]);
////			if( visible, {
////				bndl.add([ '/methodr', '[', '/method', this.id, \getParent, ']', \repaint ]);
////			}, {
////				tempID = server.nextNodeID;
////				bndl.add([ '/local', tempID, '[', '/new', "javax.swing.JPanel", ']' ]);
////				bndl.add([ '/set', this.prGetWindow.id, \contentPane, '[', '/ref', tempID, ']' ]);
////				bndl.add([ '/method', tempID, \revalidate ]);
////				bndl.add([ '/set', this.prGetWindow.id, \contentPane, '[', '/ref', this.id, ']' ]);
////			});
//////			bndl.add([ '/methodr', '[', '/method', this.prGetWindow.id, \getRootPane, ']', \repaint ]);
////			bndl.add([ '/methodr', '[', '/method', cntID, \getParent, ']', \validate ]);
////			bndl.add([ '/methodr', '[', '/method', cntID, \getParent, ']', \repaint ]);
//			bndl.add([ '/method', this.id, \repaint ]);
//			server.listSendBundle( nil, bndl );
//		});
//	}

	// ----------------- private class methods -----------------

	*new { arg window, bounds, id;
		^super.new.prInitTopView( window, bounds, id );
	}
	
	// ----------------- public instance methods -----------------

	defaultReceiveDrag {
		var win, view;
		win = this.findWindow;
		view = currentDrag.paletteExample( win, Rect( 10, 10, 140, 24 ));
		view.keyDownAction_({ arg view, char, modifiers, unicode, keycode;
			if( keycode == 51, { view.remove });
		});
	}

	// ----------------- private instance methods -----------------

	prChildOrder { ^0 }
	
	init { }	// kind of overriden by prInitTopView

	prInitTopView { arg argWindow, argBounds, id;
//		parent		= argParent.asView;	// actual view
		window		= argWindow;
//		scBounds		= argBounds;
//		jBounds		= this.prBoundsToJava( scBounds );
//		jinsets		= Insets.new;
		this.prInit( nil, argBounds.asRect, this.class.viewClass, window.server, id );
//		argParent.add( this );		// maybe window or viewadapter
	}

	prSCViewNew { arg preMsg, postMsg;
		var bndl, argBounds;
		
		if( jinsets.isNil, { jinsets = Insets.new });
		
		bndl			= List.new;
		bndl.addAll( preMsg );
		jBounds		= this.prBoundsToJava( scBounds );
//		argBounds		= jBounds.asSwingArg;
//		bndl.add([ '/set', this.id, \bounds ] ++ argBounds ++ [ \font, '[', '/ref', \font, ']' ]);
//		if( this.prIsInsideContainer, {
//			bndl.add([ '/set', "cn" ++ this.id, \bounds ] ++ argBounds );
//		});
		if( this.prNeedsTransferHandler, {
			this.prCreateDnDResponder( bndl );
		});
		// NOTE: for global key actions to be working, every view
		// has to create a key responder, even if it's not using it personally ;-(
		this.prCreateKeyResponder( bndl );
		this.prCreateCompResponder( bndl );
		bndl.addAll( postMsg );
		server.listSendBundle( nil, bndl );
	}

	prGetWindow { ^window }

	// created by JSCWindow
	handleKeyDownBubbling { arg view, char, modifiers, unicode, keycode;
		keyDownAction.value( view, char, modifiers, unicode, keycode );
	}

	handleKeyUpBubbling { arg view, char, modifiers, unicode, keycode;
		keyUpAction.value( view, char, modifiers, unicode, keycode );
	}

	prBoundsToJava { arg rect; ^rect.copy }
	prBoundsFromJava { arg rect; ^rect.copy }

	prBoundsUpdated {
		if( window.drawHook.notNil, { window.refresh });
	}
	
	prAllVisible {
		^(visible and: { this.prGetWindow.visible });
	}
}

JSCScrollTopView : JSCTopView {
	var <autohidesScrollers = true, <hasHorizontalScroller = true, <hasVerticalScroller = true;
	var <autoScrolls = true;
	var vpID, chResp;
	
	var viewX = 0, viewY = 0, viewW = 0, viewH = 0;
	
	// ----------------- public instance methods -----------------

	autohidesScrollers_ { arg bool;
		var hPolicy, vPolicy;
		autohidesScrollers = bool;
		hPolicy = JSCScrollView.protCalcPolicy( bool, hasHorizontalScroller ) + 30;
		vPolicy = JSCScrollView.protCalcPolicy( bool, hasVerticalScroller ) + 20;

		server.sendMsg( '/set', this.id, \horizontalScrollBarPolicy, hPolicy, \verticalScrollBarPolicy, vPolicy );
	}
	
	hasHorizontalScroller_ { arg bool;
		var policy;
		hasHorizontalScroller = bool;
		policy = JSCScrollView.protCalcPolicy( autohidesScrollers, bool ) + 30;
		server.sendMsg( '/set', this.id, \horizontalScrollBarPolicy, policy );
	}
	
	hasVerticalScroller_ { arg bool;
		var policy;
		hasVerticalScroller = bool;
		policy = JSCScrollView.protCalcPolicy( autohidesScrollers, bool ) + 20;
		server.sendMsg( '/set', this.id, \verticalScrollBarPolicy, policy );
	}
	
	visibleOrigin_ { arg point;
		viewX	= point.x;
		viewY	= point.y;
		server.sendMsg( '/method', this.id, \setViewPosition, point.x, point.y );
		this.doAction;
	}
	
	visibleOrigin {
		^Point( viewX, viewY );
	}
	
	autoScrolls_ { arg bool;
		"JSCScrollTopView.autoScrolls_ : not yet implemented".warn;
		autoScrolls = bool;
//		server.sendMsg( '/set', this.id, \autoScrolls, bool );
	}
	
	innerBounds {
		^Rect( 0, 0, viewW, viewH );
	}

	// ----------------- private instance methods -----------------

	prGetRefTopLeft {
		^Point( 0, 0 );
	}

	prVisibleChild { arg pre, post, child;
		var vpID;
		if( this.prAllVisible, {
			vpID = this.prViewPortID;
			if( this.id != vpID, {
				post.add([ '/method', vpID, \validate ]);
			});
			post.add([ '/method', this.id, \revalidate ]);
			pendingValidation = false;
		}, {
			pendingValidation = true;
		});
	}

	prClose { arg preMsg, postMsg;
		chResp.remove;
		^super.prClose( preMsg ++
			[[ '/method', "ch" ++ this.id, \remove ],
		      [ '/free', "ch" ++ this.id ]], postMsg );
	}
	
	prSCViewNew { arg preMsg, postMsg;
		chResp	= OSCpathResponder( server.addr, [ '/change', this.id ], { arg time, resp, msg;
			var newVal;
		
			// [ /change, 1001, performed, viewX, ..., viewY, ..., innerWidth, ..., innerHeight, ... ]

			viewW = msg[8];
			viewH = msg[10];
			if( viewX != msg[4] or: { viewY != msg[6] }, {
				viewX = msg[4];
				viewY = msg[6];
				{ this.doAction }.defer;
			});
		}).add;
		vpID = "vp" ++ this.id;
		^super.prSCViewNew([[ '/local', vpID, '[', '/methodr', '[', '/method', this.id, \getViewport, ']', \getView, ']',
			"ch" ++ this.id,
			'[', '/new', "de.sciss.swingosc.ChangeResponder", this.id, '[', '/array', \viewX, \viewY, \innerWidth, \innerHeight, ']', ']' ]]);
	}

//	prInit { arg ... args;
//		var result;
//		result = super.prInit( *args );
//		vpID = "vp" ++ this.id;
//		server.sendMsg( '/local', vpID, '[', '/methodr', '[', '/method', this.id, \getViewport, ']', \getView, ']',
//			"ac" ++ this.id,
//			'[', '/new', "de.sciss.swingosc.ChangeResponder", this.id, '[', '/array', \viewX, \viewY, \innerWidth, \innerHeight ']', ']' );
//	}

	prViewPortID { ^vpID }
}

JSCScrollView : JSCContainerView {
	var <autohidesScrollers = true, <hasHorizontalScroller = true, <hasVerticalScroller = true;
	var <autoScrolls = true;
	var vpID, chResp;

	var viewX = 0, viewY = 0, viewW = 0, viewH = 0;
	
	var <hasBorder = false;

	// ----------------- public instance methods -----------------

	hasBorder_ { arg bool = true;
		if( hasBorder != bool, {
			hasBorder = bool;
			this.setProperty( \border, bool );
		});
	}

	autohidesScrollers_ { arg bool;
		var hPolicy, vPolicy;
		autohidesScrollers = bool;
		hPolicy = JSCScrollView.protCalcPolicy( bool, hasHorizontalScroller ) + 30;
		vPolicy = JSCScrollView.protCalcPolicy( bool, hasVerticalScroller ) + 20;

		server.sendMsg( '/set', this.id, \horizontalScrollBarPolicy, hPolicy, \verticalScrollBarPolicy, vPolicy );
	}
	
	hasHorizontalScroller_ { arg bool;
		var policy;
		hasHorizontalScroller = bool;
		policy = JSCScrollView.protCalcPolicy( autohidesScrollers, bool ) + 30;
		server.sendMsg( '/set', this.id, \horizontalScrollBarPolicy, policy );
	}
	
	hasVerticalScroller_ { arg bool;
		var policy;
		hasVerticalScroller = bool;
		policy = JSCScrollView.protCalcPolicy( autohidesScrollers, bool ) + 20;
		server.sendMsg( '/set', this.id, \verticalScrollBarPolicy, policy );
	}

	visibleOrigin_ { arg point;
		viewX	= point.x;
		viewY	= point.y;
		server.sendMsg( '/method', this.id, \setViewPosition, point.x, point.y );
		this.doAction;
	}
	
	visibleOrigin {
		^Point( viewX, viewY );
	}
	
	autoScrolls_ { arg bool;
		"JSCScrollView.autoScrolls_ : not yet implemented".warn;
		autoScrolls = bool;
//		server.sendMsg( '/set', this.id, \autoScrolls, bool );
	}
	
	innerBounds {
		^Rect( 0, 0, viewW, viewH );
	}

	// ----------------- private class methods -----------------

	*protCalcPolicy { arg auto, has;
//		autohidesScrollers			1	0	1	0
//		hasHorizontalScroller		1	1	0	0
//		--------------------------------------------
//		horizontalScrollBarPolicy	0	2	1	1	+ 30
		^(has.not.binaryValue | ((auto.not && has).binaryValue << 1));
	}
	
	// ----------------- private instance methods -----------------

	prGetRefTopLeft {
		^Point( 0, 0 );
	}

	prVisibleChild { arg pre, post, child;
		var vpID;
		if( this.prAllVisible, {
			vpID = this.prViewPortID;
			if( this.id != vpID, {
				post.add([ '/method', vpID, \validate ]);
			});
			post.add([ '/method', this.id, \revalidate ]);
			pendingValidation = false;
		}, {
			pendingValidation = true;
		});
	}

	prClose { arg preMsg, postMsg;
		chResp.remove;
		^super.prClose( preMsg ++
			[[ '/method', "ch" ++ this.id, \remove ],
		      [ '/free', "ch" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		chResp	= OSCpathResponder( server.addr, [ '/change', this.id ], { arg time, resp, msg;
			var newVal;
		
			// [ /change, 1001, performed, viewX, ..., viewY, ..., innerWidth, ..., innerHeight, ... ]

			viewW = msg[8];
			viewH = msg[10];
			if( viewX != msg[4] or: { viewY != msg[6] }, {
				viewX = msg[4];
				viewY = msg[6];
				{ this.doAction }.defer;
			});
		}).add;
		vpID = "vp" ++ this.id;
		^super.prSCViewNew([
			[ '/local', vpID, '[', '/new', "de.sciss.swingosc.ContentPane", 0, ']',
			  this.id, '[', '/new', "de.sciss.swingosc.ScrollPane", '[', '/ref', vpID, ']', ']',
 			  "ch" ++ this.id, '[', '/new', "de.sciss.swingosc.ChangeResponder", this.id,
 			  	'[', '/array', \viewX, \viewY, \innerWidth, \innerHeight, ']', ']' ]]);
	}

	prInit { arg ... args;
		var result;
		result = super.prInit( *args );
		vpID = "vp" ++ this.id;
		server.sendMsg( '/local', vpID, '[', '/methodr', '[', '/method', this.id, \getViewport, ']', \getView, ']' );
	}

	prViewPortID { ^vpID }
}

// abstract class!
JSCLayoutView : JSCContainerView {
	// ----------------- public instance methods -----------------

	spacing { ^this.getProperty( \spacing, 0 )}
	
	spacing_ { arg distance; this.setProperty( \spacing, distance )}

	// ----------------- quasi-interface methods : crucial-lib support -----------------

	asFlowView {}

	// ----------------- private instance methods -----------------

	properties { ^super.properties ++ #[ \spacing ]}

	prSendProperty { arg key, value;
		var bndl;

		key	= key.asSymbol;

		switch( key,
			\spacing, {
				server.sendBundle( nil, [ '/methodr', '[', '/method', this.id, \getLayout, ']', \setSpacing, value ],
									 [ '/method', this.id, \revalidate ]);
				^nil;
			}
		);
		^super.prSendProperty( key, value );
	}

	prSCViewNew { arg preMsg, postMsg;
		properties.put( \spacing, 4 );
		jinsets = Insets( 3, 3, 3, 3 );  // so focus borders of children are not clipped
		^super.prSCViewNew( preMsg, postMsg );
	}
}

JSCHLayoutView : JSCLayoutView {
	// ----------------- private instance methods -----------------

	prSCViewNew {
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.Panel", '[', '/new', "de.sciss.swingosc.ColliderAxisLayout", 0, 4, ']', ']' ]
		]);
	}
}

JSCVLayoutView : JSCLayoutView {
	// ----------------- private instance methods -----------------

	prSCViewNew {
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.Panel", '[', '/new', "de.sciss.swingosc.ColliderAxisLayout", 1, 4, ']', ']' ]
		]);
	}
}

JSCControlView : JSCView {} // abstract class

JSCSliderBase : JSCControlView {
	// ----------------- public instance methods -----------------

	knobColor {
		^this.getProperty(\knobColor, Color.new)
	}
	
	knobColor_ { arg color;
		this.setProperty(\knobColor, color)
	}
	
	step_ { arg stepSize;
		this.setPropertyWithAction(\step, stepSize);
	}
	step {
		^this.getProperty(\step)
	}
	
	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \knobColor, \step ];
	}

	prSnap { arg val;
		if( this.step <= 0.0, {
			^val.clip( 0.0, 1.0 );
		}, {
			^(val.clip( 0.0, 1.0 ) / this.step).round * this.step;
		});
	}
}

JSCSlider : JSCSliderBase
{
	var acResp, keyResp;	// OSCpathResponder for action listening
	var orientation;	// 0 for horiz, 1 for vert
	var clpse;

	// ----------------- public instance methods -----------------

	value { ^this.getProperty( \value )}
	
	value_ { arg val;
		this.setProperty( \value, this.prSnap( val ));
	}
	
	valueAction_ { arg val;
		this.setPropertyWithAction( \value, this.prSnap( val ));
	}	
	
	increment {
		var inc;
		inc = (if( orientation == 0, { this.bounds.width }, { this.bounds.height }) - 2).max( 1 ).reciprocal.max( this.step );
		^this.valueAction = this.value + inc;
	}
	
	decrement {
		var inc;
		inc = (if( orientation == 0, { this.bounds.width }, { this.bounds.height }) - 2).max( 1 ).reciprocal.max( this.step );
		^this.valueAction = this.value - inc;
	}
	
	thumbSize { ^this.getProperty( \thumbSize, 12 )}
	
	thumbSize_ { arg size;
		"JSCSlider.thumbSize_ : not yet implemented".warn;
		this.setProperty( \thumbSize, size );
	}
	
	bounds_ { arg rect;
		var result;
		result = super.bounds_( rect );
		if( if( rect.width > rect.height, 0, 1 ) != orientation, {
			orientation = 1 - orientation;
			server.sendMsg( '/set', this.id, \orientation, orientation );
		});
		^result;
	}

	defaultKeyDownAction { arg char, modifiers, unicode, keycode;
		// standard keydown
		if (char == $r, { this.valueAction = 1.0.rand; ^this });
		if (char == $n, { this.valueAction = 0.0; ^this });
		if (char == $x, { this.valueAction = 1.0; ^this });
		if (char == $c, { this.valueAction = 0.5; ^this });
		if (char == $], { this.increment; ^this });
		if (char == $[, { this.decrement; ^this });
		if (unicode == 16rF700, { this.increment; ^this });
		if (unicode == 16rF703, { this.increment; ^this });
		if (unicode == 16rF701, { this.decrement; ^this });
		if (unicode == 16rF702, { this.decrement; ^this });
		^nil		// bubble if it's an invalid key
	}
	
	defaultCanReceiveDrag { ^currentDrag.isNumber }
	defaultGetDrag { ^this.value }
	defaultReceiveDrag { this.valueAction = currentDrag }

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \thumbSize ];
	}

	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		clpse.cancel;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		properties.put( \value, 0.0 );
		properties.put( \step, 0.0 );
		orientation = if( this.bounds.width > this.bounds.height, 0, 1 );
		clpse	= Collapse({ this.doAction });
		acResp	= OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var newVal;
		
//			newVal = msg[4] / 0x40000000;
			newVal = this.prSnap( msg[4] / 0x40000000 );
			if( newVal != this.value, {
				// don't call valueAction coz we'd create a loop
				properties.put( \value, newVal );
				clpse.instantaneous;
			});
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id,
				'[', '/new', "de.sciss.swingosc.Slider", orientation, 0, 0x40000000, 0, ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, \value, ']' ]
		]);
	}
	
	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case { key === \value }
		{
			key		= \valueNoAction;
			value	= value * 0x40000000;
		}
		{ key === \step }
		{
			value = max( 1, value * 0x40000000 ).asInteger;
//			server.sendMsg( '/set', this.id, \snapToTicks, value != 0,
//							\minorTickSpacing, value, \extent, value );
			server.sendMsg( '/set', this.id, \snapToTicks, value != 0,
							\majorTickSpacing, value ); // stupidly, using extent won't let you move the slider to the max
			^nil;
		};
		^super.prSendProperty( key, value );
	}
}

JSCKnob : JSCSlider {}

JSCRangeSlider : JSCSliderBase {

	var acResp;	// OSCpathResponder for action listening
	var clpse;
	var orientation;	// 0 for horiz, 1 for vert

	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		var v;
		v = this.new( parent, bounds );
		v.setSpan( 0.2, 0.7 );
		^v;
	}

	// ----------------- public instance methods -----------------

	step_ { arg stepSize;
		super.step_( stepSize );
		this.setSpan( this.lo, this.hi );
	}
	
	lo { ^this.getProperty( \lo )}

	lo_ { arg val;
		this.setProperty( \lo, this.prSnap( val ));
	}
	
	activeLo_ { arg val;
		this.setPropertyWithAction( \lo, this.prSnap( val ));
	}
	
	hi { ^this.getProperty( \hi )}

	hi_ { arg val;
		this.setProperty( \hi, this.prSnap( val ));
	}
	
	activeHi_ { arg val;
		this.setPropertyWithAction( \hi, this.prSnap( val ));
	}
	
	range { ^(this.hi - this.lo).abs }

	range_ { arg val;
		this.hi_( this.prSnap( this.lo + val ));
	}
	
	activeRange_ { arg val;
		this.range_( val );
		this.doAction;
	}
	
	setSpan { arg lo, hi;
		lo = this.prSnap( lo );
		hi = this.prSnap( hi );
		properties.put( \lo, lo );
		properties.put( \hi, hi );
		server.sendMsg( '/set', this.id, \knobPos, min( lo, hi ), \knobExtent, abs( hi - lo ));
	}
	
	setSpanActive { arg lo, hi;
		this.setSpan( lo, hi );
		this.doAction;
	}

	increment {
		var inc, val; 
		inc = (if( orientation == 0, { this.bounds.width }, { this.bounds.height }) - 2).max( 1 ).reciprocal;
		val = this.hi + inc;
		if( val > 1, {
			inc = 1 - this.hi;
			val = 1;
		});
		this.setSpanActive( this.lo + inc, val );
	}
	
	decrement { 
		var inc, val; 
		inc = (if( orientation == 0, { this.bounds.width }, { this.bounds.height }) - 2).max( 1 ).reciprocal;
		val = this.lo - inc;
		if( val < 0, {
			inc = this.lo;
			val = 0;
		});
		this.setSpanActive( val, this.hi - inc );
	}

	bounds_ { arg rect;
		var result;
		result = super.bounds_( rect );
		if( if( rect.width > rect.height, 0, 1 ) != orientation, {
			orientation = 1 - orientation;
			server.sendMsg( '/set', this.id, \orientation, orientation );
		});
		^result;
	}

	defaultKeyDownAction { arg char, modifiers, unicode;
		var a, b;
		// standard keydown
		if (char == $r, { 
			a = 1.0.rand;
			b = 1.0.rand;
			this.setSpanActive( min( a, b ), max( a, b ));
			^this;
		});
		if (char == $n, { this.setSpanActive( 0.0, 0.0 ); ^this });
		if (char == $x, { this.setSpanActive( 1.0, 1.0 ); ^this });
		if (char == $c, { this.setSpanActive( 0.5, 0.5 ); ^this });
		if (char == $a, { this.setSpanActive( 0.0, 1.0 ); ^this });
		if (unicode == 16rF700, { this.increment; ^this });
		if (unicode == 16rF703, { this.increment; ^this });
		if (unicode == 16rF701, { this.decrement; ^this });
		if (unicode == 16rF702, { this.decrement; ^this });
		^nil;		// bubble if it's an invalid key
	}

	defaultGetDrag { ^Point( this.lo, this.hi )}	
	defaultCanReceiveDrag {	 ^currentDrag.isKindOf( Point )}
	
	defaultReceiveDrag {
		// changed to x,y instead of lo, hi
		this.setSpanActive( currentDrag.x, currentDrag.y );
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \lo, \hi ];
	}
	
	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		clpse.cancel;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ],
			 [ '/method', this.id, \dispose ]], postMsg );
	}

	prSCViewNew {
		properties.put( \lo, 0.0 );
		properties.put( \hi, 1.0 );
		properties.put( \step, 0.0 );
		jinsets		= Insets( 3, 3, 3, 3 );
		orientation	= if( this.bounds.width > this.bounds.height, 0, 1 );
		clpse		= Collapse({ this.doAction });
		acResp		= OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var newLo, newHi;
		
			newLo	= msg[4];
			newHi 	= newLo + msg[6];
			if( (newLo != this.lo) || (newHi != this.hi), {
				// don't call valueAction coz we'd create a loop
				properties.put( \lo, newLo );
				properties.put( \hi, newHi );
				clpse.instantaneous;
			});
		}).add;
		^super.prSCViewNew([
			[ '/set', '[', '/local', this.id, '[', '/new', "de.sciss.swingosc.RangeSlider", orientation, ']', ']',
				\knobColor ] ++ Color.blue.asSwingArg,
			[ '/local', "ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, '[', '/array', \knobPos, \knobExtent, ']', ']' ]
		]);
	}

	prSendProperty { arg key, value;
		key	= key.asSymbol;

		// fix keys
		case { key === \lo }
		{
			server.sendMsg( '/set', this.id, \knobPos, min( value, this.hi ), \knobExtent, abs( value - this.hi ));
			^nil;		
		}
		{ key === \hi }
		{
			server.sendMsg( '/set', this.id, \knobPos, min( value, this.lo ), \knobExtent, abs( value - this.lo ));
			^nil;		
		}
		{ key === \step }
		{
			key = \stepSize;
		};
		^super.prSendProperty( key, value );
	}
}

JSC2DSlider : JSCSliderBase {

	var acResp;	// OSCpathResponder for action listening
	var clpse;

	// ----------------- public instance methods -----------------

	step_ { arg stepSize;
		super.step_( stepSize );
		this.x_( this.x );
		this.y_( this.y );
	}

	x { ^this.getProperty( \x )}

	x_ { arg val;
		this.setProperty( \x, this.prSnap( val ));
	}
	
	activex_ { arg val;
		this.setPropertyWithAction( \x, this.prSnap( val ));
	}
	
	y { ^this.getProperty( \y )}

	y_ { arg val;
		this.setProperty( \y, this.prSnap( val ));
	}
	
	activey_ { arg val;
		this.setPropertyWithAction( \y, this.prSnap( val ));
	}
	
	setXY { arg x, y;
		x = this.prSnap( x );
		y = this.prSnap( y );
		properties.put( \x, x );
		properties.put( \y, y );
		server.sendMsg( '/set', this.id, \knobX, x, \knobY, y );
	}
	
	setXYActive { arg x, y;
		this.setXY( x, y );
		this.doAction;
	}

	incrementY { ^this.y = this.y + this.bounds.height.reciprocal }
	decrementY { ^this.y = this.y - this.bounds.height.reciprocal }
	incrementX { ^this.x = this.x + this.bounds.width.reciprocal }
	decrementX { ^this.x = this.x - this.bounds.width.reciprocal }

	defaultKeyDownAction { arg char, modifiers, unicode,keycode;
		// standard keydown
		if (char == $r, { this.setXYActive( 1.0.rand, 1.0.rand ); ^this });
		if (char == $n, { this.setXYActive( 0.0, 0.0 ); ^this });
		if (char == $x, { this.setXYActive( 1.0, 1.0 ); ^this });
		if (char == $c, { this.setXYActive( 0.5, 0.5 ); ^this });
		if (unicode == 16rF700, { this.incrementY; this.doAction; ^this });
		if (unicode == 16rF703, { this.incrementX; this.doAction; ^this });
		if (unicode == 16rF701, { this.decrementY; this.doAction; ^this });
		if (unicode == 16rF702, { this.decrementX; this.doAction; ^this });
		^nil		// bubble if it's an invalid key
	}

	defaultGetDrag { ^Point( this.x, this.y )}
	defaultCanReceiveDrag { ^currentDrag.isKindOf( Point )}
	defaultReceiveDrag { this.setXYActive( currentDrag.x, currentDrag.y )}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \x, \y ];
	}
	
	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		clpse.cancel;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ],
			 [ '/method', this.id, \dispose ]], postMsg );
	}

	prSCViewNew {
		properties.put( \x, 0.0 );
		properties.put( \y, 0.0 );
		properties.put( \step, 0.0 );
		jinsets	= Insets( 3, 3, 3, 3 );
		clpse	= Collapse({ this.doAction });
		acResp	= OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var newX, newY;
		
			newX = msg[4];
			newY = msg[6];
			if( (newX != this.x) || (newY != this.y), {
				// don't call valueAction coz we'd create a loop
				properties.put( \x, newX );
				properties.put( \y, newY );
				clpse.instantaneous;
			});
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id,
				'[', '/new', "de.sciss.swingosc.Slider2D", ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, '[', '/array', \knobX, \knobY, ']', ']' ]
		]);
	}

	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case { key === \x }
		{
			key = \knobX;
		}
		{ key === \y }
		{
			key = \knobY;
		}
		{ key === \step }
		{
			key = \stepSize;
		};
		^super.prSendProperty( key, value );
	}
}

//// JJJ : not yet working
//JSC2DTabletSlider : JSC2DSlider {
//
////	var <>mouseDownAction,<>mouseUpAction;
//	
//	mouseDown { arg x,y,pressure,tiltx,tilty,deviceID,
//			 buttonNumber,clickCount,absoluteZ,rotation;
//		mouseDownAction.value(this,x,y,pressure,tiltx,tilty,deviceID, 
//			buttonNumber,clickCount,absoluteZ,rotation);
//	}
//	mouseUp { arg x,y,pressure,tiltx,tilty,deviceID, 
//			buttonNumber,clickCount,absoluteZ,rotation;
//		mouseUpAction.value(this,x,y,pressure,tiltx,tilty,deviceID, 
//			buttonNumber,clickCount,absoluteZ,rotation);
//	}
//	doAction { arg x,y,pressure,tiltx,tilty,deviceID, 
//			buttonNumber,clickCount,absoluteZ,rotation;
//		action.value(this,x,y,pressure,tiltx,tilty,deviceID, 
//			buttonNumber,clickCount,absoluteZ,rotation);
//	}
//}

JSCButton : JSCControlView {
	var <states;
	
	var acResp;	// OSCpathResponder for action listening

	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		var v;
		v = this.new( parent, bounds );
		v.states = [
			[ "Push", Color.black, Color.red ],
			[ "Pop", Color.white, Color.blue ]];
		^v;
	}
	
	// ----------------- public instance methods -----------------

	value { ^this.getProperty( \value )}
	
	value_ { arg val;
		this.setProperty( \value, this.prFixValue( val ));
	}
	
	valueAction_ { arg val;
		this.setPropertyWithAction( \value, this.prFixValue( val ));
	}	

	doAction { arg modifiers;
		action.value( this, modifiers );
	}
	
	font { ^this.getProperty( \font )}

	font_ { arg argFont;
//		font = argFont;
		this.setProperty( \font, argFont );
	}

	states_ { arg array;
		states = array.deepCopy;
		this.setProperty( \states, states );
	}
	
	defaultKeyDownAction { arg char, modifiers, unicode;
// JJJ handled automatically by javax.swing.AbstractButton
//		if (char == $ , { this.valueAction = this.value + 1; ^this });
		if (char == $\r, { this.valueAction = this.value + 1; ^this });
		if (char == $\n, { this.valueAction = this.value + 1; ^this });
		if (char == 3.asAscii, { this.valueAction = this.value + 1; ^this });
		^nil;		// bubble if it's an invalid key
	}

	defaultGetDrag { ^this.value }

	defaultCanReceiveDrag {
		^currentDrag.isNumber or: { currentDrag.isKindOf( Function )};
	}
	
	defaultReceiveDrag {
		if( currentDrag.isNumber, {
			this.valueAction = currentDrag;
		}, {
			this.action = currentDrag;
		});
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \value, \font, \states ];
	}
	
	prFixValue { arg val;
		val = val.asInteger;
		// clip() would be better but SCButton resets to zero always
		if( (val < 0) || (val >= states.size), {
			val = 0;
		});
		^val;
	}
	
	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		properties.put( \value, 0 );
		jinsets = Insets( 3, 3, 3, 3 );
		acResp = OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var value, modifiers;
			value	= msg[4];
			modifiers	= msg[6];
			// java->cocoa ; this translates shift (1), ctrl (2), cmd (4), alt (8)
//			modifiers		= ((modifiers & 3) << 17) |
//						  ((modifiers & 4) << 18) |
//						  ((modifiers & 8) << 16); // | plusMod;
			// don't call valueAction coz we'd create a loop
			properties.put( \value, msg[4] );
			{ this.doAction( modifiers )}.defer;
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.gui.MultiStateButton", ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, '[', '/array', \selectedIndex, \lastModifiers, ']', ']' ]
//				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, '[', '/array', \selectedIndex, ']', ']' ]
		]);
	}

	prSendProperty { arg key, value;
		var bndl, msg;

		key	= key.asSymbol;

		// fix keys
		case { key === \value }
		{
			key = \selectedIndex;
		}
		{ key === \states }
		{
			bndl	= List.new;
			bndl.add([ '/method', this.id, \removeAllItems ]);
			value.do({ arg state;
				msg = List.newUsing([ '/method', this.id, \addItem ]);
				msg.addAll( state[0].asSwingArg );
				msg.addAll( state[1].asSwingArg );
				msg.addAll( state[2].asSwingArg );
				bndl.add( msg.asArray );
			});
//("Sending '"++bndl++"'").postln;
			server.listSendBundle( nil, bndl );
			^nil;
		};
		^super.prSendProperty( key, value );
	}
}


JSCPopUpMenu : JSCControlView {
	var <items;
	
	var acResp;	// OSCpathResponder for action listening
	var <>allowsReselection = false;

	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		var v;
		v = this.new(parent, bounds);
		v.items = #[ "linear", "exponential", "sine", "welch", "squared", "cubed" ];
		^v;
	}
		
	// ----------------- public instance methods -----------------

	value { ^this.getProperty( \value )}
	
	value_ { arg val;
		this.setProperty( \value, this.prFixValue( val ));
	}
		
	valueAction_ { arg val;
		this.setPropertyWithAction( \value, this.prFixValue( val ));
	}
	
	font { ^this.getProperty( \font )}
	
	font_ { arg argFont;
//		font = argFont;
		this.setProperty( \font, argFont );
	}
	
	items_ { arg array;
		items = array.copy;
		this.setProperty( \items, items );
	}
	
	item { ^items[ this.value ]}

	stringColor {
		^this.getProperty( \stringColor, Color.new );
	}
	
	stringColor_ { arg color;
		this.setProperty( \stringColor, color );
	}
	
	defaultKeyDownAction { arg char, modifiers, unicode;
// JJJ used by lnf
//		if (char == $ , { this.valueAction = this.value + 1; ^this });
//		if (char == $\r, { this.valueAction = this.value + 1; ^this });
//		if (char == $\n, { this.valueAction = this.value + 1; ^this });
		if (char == 3.asAscii, { this.valueAction = this.value + 1; ^this });
//		if (unicode == 16rF700, { this.valueAction = this.value - 1; ^this });
		if (unicode == 16rF703, { this.valueAction = this.value + 1; ^this });
//		if (unicode == 16rF701, { this.valueAction = this.value + 1; ^this });
		if (unicode == 16rF702, { this.valueAction = this.value - 1; ^this });
		^nil		// bubble if it's an invalid key
	}
	
	defaultGetDrag { ^this.value }
	defaultCanReceiveDrag { ^currentDrag.isNumber }

	defaultReceiveDrag {
		this.valueAction = currentDrag;
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \value, \font, \items, \stringColor ];
	}

	prFixValue { arg val;
		^val.clip( 0, items.size - 1 );
	}

	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		properties.put( \value, 0 );
		acResp = OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var newVal;
			
			newVal = msg[4];
			if( allowsReselection or: { newVal != this.value }, {
				// don't call valueAction coz we'd create a loop
				properties.put( \value, newVal );
				{ this.doAction; }.defer;
			});
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.PopUpView", ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionResponder", this.id, \selectedIndex, ']' ]
		]);
	}

	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case
//		{ key === \value }
//		{
//			key = \selectedIndex;
//		}
		{ key === \items }
		{
			this.prSetItems( value.performUnaryOp( \asString ));
			^nil;
		};
		^super.prSendProperty( key, value );
	}

// XXX at the moment...
//	prBoundsToJava { arg rect;
//		var pb;
//		
//		pb = parent.bounds;
//		if( rect.height < 26, {
//			rect			= Rect( rect.left - pb.left, rect.top - ((26 - rect.height) >> 1) - pb.top,
//							   rect.width, 26 );
//		}, {
//			rect	= rect.moveBy( pb.left.neg, pb.top.neg );
//		});
//		^rect;
//	}

// what shall we do ...
//	prBoundsFromJava { arg rect;
//		^rect;
//	}

	prSetItems { arg items;
		var sizes, dataSize, startIdx, bndl, val;

		sizes	= items.collect({ arg item; ((item.size + 4) & -4) + 1 });
		val		= this.value;
		if( val >= items.size, { val = -1 });
//("sum = "++sizes.sum).postln;
		if( (sizes.sum + 70) < server.options.oscBufSize, {
			server.listSendMsg([ '/method', this.id, \setListData, '[', '/array' ] ++ items ++ [ ']', val ]);
		}, {	// need to split it up
			startIdx = 0;
//			dataSize	= 45;
//			server.sendMsg( '/method', this.id, \beginDataUpdate );
			dataSize	= 171; // 139 (there seems to be a limit below 64K)
			bndl		= Array( 3 );
			bndl.add([ '/method', this.id, \beginDataUpdate ]);
			sizes.do({ arg size, idx;
				if( (dataSize + size) > server.options.oscBufSize, {
("sending : "++dataSize).postln;
//					server.listSendMsg([ '/method', this.id, \addData, '[', '/array', ] ++
//						items.copyRange( startIdx, idx - 1 ) ++ [ ']' ]);
					bndl.add([ '/method', this.id, \addData, '[', '/array', ] ++
						items.copyRange( startIdx, idx - 1 ) ++ [ ']' ]);
					server.listSendBundle( nil, bndl );
//					dataSize	= 45;
					startIdx	= idx;
					bndl		= Array( 2 );
					dataSize	= 135; // 103 (there seems to be a limit below 64K)
				}, {
					dataSize = dataSize + size;
				});
			});
//			server.listSendMsg([ '/method', this.id, \addData, '[', '/array', ] ++
//					items.copyRange( startIdx, items.size - 1 ) ++ [ ']' ]);
//			server.sendMsg( '/method', this.id, \endDataUpdate, this.value );
			bndl.add([ '/method', this.id, \addData, '[', '/array', ] ++
					items.copyRange( startIdx, items.size - 1 ) ++ [ ']' ]);
			bndl.add([ '/method', this.id, \endDataUpdate, val ]);
			server.listSendBundle( nil, bndl );
			([ "sending : " ]++ bndl).postcs;
		});
	}
}

JSCStaticTextBase : JSCView {
	var <string, <object, <>setBoth = true;
	
	// ----------------- public instance methods -----------------

	font { ^this.getProperty( \font )}
	
	font_ { arg argFont;
//		font = argFont;
		this.setProperty( \font, argFont );
	}
	
	string_ { arg argString;
		string = argString.asString;
		this.setProperty(\string, string)
	}
	align_ { arg align;
		this.setProperty(\align, align)
	}
	
	stringColor {
		^this.getProperty(\stringColor, Color.new)
	}
	stringColor_ { arg color;
		this.setProperty(\stringColor, color)
	}

	object_ { arg obj;
		object = obj;
		if( setBoth, { this.string = object.asString( 80 )});
	}
	
	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \string, \font, \stringColor ];
	}

	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case { key === \stringColor }
		{
			key = \foreground;
		}
		{ key === \align }
		{
			key = \horizontalAlignment;
			case { value === \left }
			{
				value = 2;
			}
			{ value === \center }
			{
				value = 0;
			}
			{ value === \right }
			{
				value = 4;
			}
			// undocumented cocoa feature : -1 = left, 0 = center, 1 = right
			{ value.isKindOf( SimpleNumber )}
			{
				value = switch( value.sign, -1, 2, 0, 0, 1, 4 );
			};
		}
		{ key === \string }
		{
			key = \text;
//			value = value.asSwingArg;
// funktioniert nicht, weil BoundedRangeModel offensichtlich nochmal durch setText veraendert wird
//			server.sendBundle( nil,
//				[ '/set', this.id, \text, value ],	// make sure the text beginning is shown
//				[ "/methodr", [ '/method', this.id, \getHorizontalVisibility ], \setValue, 0 ]
//			);
//			^nil;
		};
		^super.prSendProperty( key, value );
	}
}

JSCStaticText : JSCStaticTextBase {

	// ----------------- public class methods -----------------
	
	*paletteExample { arg parent, bounds;
		var v;
		v = this.new(parent, bounds);
		v.string = "The lazy brown fox";
		^v
	}

	// ----------------- private instance methods -----------------

	prSCViewNew {
		properties.put( \canFocus, false );
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.Label", ']' ]
		]);
	}
}

JSCListView : JSCControlView {
	var <items, <>enterKeyAction;
	var <allowsDeselection = false;
	
	var acResp;	// listens to list selection changes
	var cnID;
	
	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		var v;
		v = this.new(parent, bounds);
		v.items = #[ "linear", "exponential", "sine", "welch", "squared", "cubed" ];
		^v;
	}
	
	// ----------------- public instance methods -----------------

	item { ^items[ this.value ]}

	value { ^this.getProperty( \value )}

	value_ { arg val;
		this.setProperty( \value, this.prFixValue( val ));
	}
	
	valueAction_ { arg val;
		this.setPropertyWithAction( \value, this.prFixValue( val ));
	}
	
	allowsDeselection_ { arg bool;
		if( allowsDeselection != bool, {
			allowsDeselection = bool;
			if( allowsDeselection, {
				if( (this.value == 0) and: { items.size == 0 }, {
					this.valueAction_( nil );
				});
			}, {
				if( this.value.isNil, {
					this.valueAction_( 0 );
				});
			});
		});
	}

	font { ^this.getProperty( \font )}
	
	font_ { arg argFont;
//		font = argFont;
		this.setProperty( \font, argFont );
	}
	
	items_ { arg array;
		items = array.copy;
		this.setProperty( \items, items );
	}
	
	stringColor {
		^this.getProperty( \stringColor, Color.new );
	}
	
	stringColor_ { arg color;
		this.setProperty( \stringColor, color );
	}
	
	selectedStringColor {
		^this.getProperty( \selectedStringColor, Color.new );
	}
	
	selectedStringColor_ { arg color;
		this.setProperty( \selectedStringColor, color );
	}
	
	hiliteColor {
		^this.getProperty( \hiliteColor, Color.new );
	}
	
	hiliteColor_ { arg color;
		this.setProperty( \hiliteColor, color );
	}
	
	defaultKeyDownAction { arg char, modifiers, unicode;
		var index;
		if( this.value.notNil, {
			if( char == $ , { this.valueAction = this.value + 1; ^this });
			if( char == $\r, { this.enterKeyAction.value(this); ^this });
			if( char == $\n, { this.enterKeyAction.value(this); ^this });
			if( char == 3.asAscii, { this.enterKeyAction.value(this); ^this });
	// JJJ automatically handled by lnf
	//		if( unicode == 16rF700, { this.valueAction = this.value - 1; ^this });
			if( unicode == 16rF703, { this.valueAction = this.value + 1; ^this });
	//		if( unicode == 16rF701, { this.valueAction = this.value + 1; ^this });
			if( unicode == 16rF702, { this.valueAction = this.value - 1; ^this });
		});
		if (char.isAlpha, {
			char = char.toUpper;
			index = items.detectIndex({ arg item; item.asString.at(0).toUpper >= char });
			if( index.notNil, { this.valueAction = index });
			^this;
		});
		^nil;	// bubble if it's an invalid key
	}
	

	defaultGetDrag { ^this.value }
	defaultCanReceiveDrag { ^currentDrag.isNumber }

	defaultReceiveDrag {
		this.valueAction = currentDrag;
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \value, \font, \items, \stringColor ];
	}

	prFixValue { arg val;
		if( allowsDeselection and: { val.isNil }, { ^nil });
		val = (val ? 0).asInteger;
		if( (val < 0) || (val >= items.size), {
			val = 0;
		});
		^val;
	}

	prNeedsTransferHandler { ^true }

	prClose { arg preMsg, postMsg;
		acResp.remove;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prContainerID { ^cnID }

	prSCViewNew {
		cnID = "cn" ++this.id;
		properties.put( \value, 0 );
		
		acResp = OSCpathResponder( server.addr, [ '/list', this.id ], { arg time, resp, msg;
			var newVal;

			newVal = this.prFixValue( if( msg[4] >= 0, msg[4] ));
			if( newVal != this.value, {
				// don't call valueAction coz we'd create a loop
				properties.put( \value, newVal );
				{ this.doAction; }.defer;
			});
		}).add;
		^super.prSCViewNew([
			[ '/set', '[', '/local', this.id, '[', '/new', "de.sciss.swingosc.ListView", ']', ']',
				\selectionMode, 0 ],	// single selection only for compatibility
			[ '/local', "cn" ++ this.id, 	// bars : v=asNeeded, h=never
				'[', '/new', "javax.swing.JScrollPane", '[', '/ref', this.id, ']', 20, 31, ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ListResponder", this.id,
					'[', '/array', \selectedIndex, ']', ']' ] // , \valueIsAdjusting
		]);
	}

	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case { key === \value }
		{
			value = value ? -1;
			if( value >= 0, {
				server.sendBundle( nil,
					[ '/set', this.id, \value, value ],
					[ '/method', this.id, \ensureIndexIsVisible, value ]
				);
			}, {
				server.sendMsg( '/set', this.id, \value, value );
			});
			^nil;
		}
		{ key === \stringColor }
		{
			key = \foreground;
		}
		{ key === \selectedStringColor }
		{
			key = \selectionForeground;
		}
		{ key === \hiliteColor }
		{
			key = \selectionBackground;
		}
		{ key === \items }
		{
			this.prSetItems( value.performUnaryOp( \asString ));
			^nil;
//		}
//		{ key === \bounds }
//		{
//			server.listSendMsg([ '/set', "cn" ++ this.id, key ] ++ this.prBoundsToJava( value ).asSwingArg );
//			^nil;
		};
		^super.prSendProperty( key, value );
	}

	prSetItems { arg items;
		var sizes, dataSize, startIdx;

		sizes = items.collect({ arg item; ((item.size + 4) & -4) + 1 });
//("sum = "++sizes.sum).postln;
		if( (sizes.sum + 70) < server.options.oscBufSize, {
			server.listSendMsg([ '/method', this.id, \setListData, '[', '/array' ] ++ items ++
											[ ']', this.value ? -1 ]);
		}, {	// need to split it up
			startIdx = 0;
			dataSize	= 45;
			server.sendMsg( '/method', this.id, \beginDataUpdate );
			sizes.do({ arg size, idx;
				if( (dataSize + size) > server.options.oscBufSize, {
//("sending : "++dataSize).postln;
					server.listSendMsg([ '/method', this.id, \addData, '[', '/array', ] ++
						items.copyRange( startIdx, idx - 1 ) ++ [ ']' ]);
					dataSize	= 45;
					startIdx	= idx;
				}, {
					dataSize = dataSize + size;
				});
			});
			server.listSendMsg([ '/method', this.id, \addData, '[', '/array', ] ++
					items.copyRange( startIdx, items.size - 1 ) ++ [ ']' ]);
			server.sendMsg( '/method', this.id, \endDataUpdate, this.value ? -1 );
		});
	}
}

JSCDragView : JSCStaticTextBase {
	var <>interpretDroppedStrings = true;
	
	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		var v;
		v = this.new(parent, bounds);
		v.object = \something;
		^v
	}
	
	// ----------------- private instance methods -----------------

	defaultGetDrag { ^object }

	prNeedsTransferHandler { ^true }

	prImportDrag {
		if( interpretDroppedStrings, { JSCView.importDrag });
	}

	prSCViewNew { arg preMsg, postMsg;
		properties.put( \canFocus, false );
		jinsets = Insets( 3, 3, 3, 3 );
		^super.prSCViewNew( preMsg, postMsg );
	}
}

JSCDragSource : JSCDragView {
	// ----------------- private instance methods -----------------

	prSCViewNew {
		^super.prSCViewNew([
//			[ '/set', '[', '/local', this.id, '[', '/new', "de.sciss.swingosc.Label", ']', ']',
//				\border, '[', '/method', "javax.swing.BorderFactory", \createRaisedBevelBorder, ']'
//			]
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.DragView", 1, ']',
			]
		]);
	}

	prGetDnDModifiers { ^0 } 	// no modifiers needed
}

JSCDragSink : JSCDragView {
	// ----------------- public instance methods -----------------

	defaultCanReceiveDrag { ^true }

	defaultReceiveDrag {
		this.object = currentDrag;
		this.doAction;
	}

	// ----------------- private instance methods -----------------

	prSCViewNew {
		^super.prSCViewNew([
//			[ '/set', '[', '/local', this.id, '[', '/new', "de.sciss.swingosc.Label", ']', ']',
//				\border, '[', '/method', "javax.swing.BorderFactory", \createLoweredBevelBorder, ']'
//			]
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.DragView", 0, ']',
			]
		]);
	}

	prGetDnDModifiers { ^-1 }	// don't allow it to be drag source
}

JSCDragBoth : JSCDragView {		// in SwingOSC not subclass of JSCDragSink
	// ----------------- public instance methods -----------------

	defaultCanReceiveDrag { ^true }
	
	defaultReceiveDrag {
		this.object = currentDrag;
		this.doAction;
	}

	defaultGetDrag { ^object }

	// ----------------- private instance methods -----------------

	prSCViewNew {
		^super.prSCViewNew([
//			[ '/set', '[', '/local', this.id, '[', '/new', "de.sciss.swingosc.Label", ']', ']',
//				\border, '[', '/method', "javax.swing.BorderFactory", \createCompoundBorder,
//					'[', '/method', "javax.swing.BorderFactory", \createRaisedBevelBorder, ']',
//					'[', '/method', "javax.swing.BorderFactory", \createLoweredBevelBorder, ']', ']'
//			]
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.DragView", 2, ']',
			]
		]);
	}

	prGetDnDModifiers { ^0 } 	// no modifiers needed
}

JSCAbstractUserView : JSCView {
	var <drawFunc;
	var <clearOnRefresh = true;
	var <>refreshOnFocus = true;
	var <relativeOrigin;

	var penID			= nil;
	var pendingDraw	= false;

	// ----------------- public instance methods -----------------

	refresh {
		pendingDraw = false;
		if( drawFunc.notNil, {
			JPen.protRefresh( drawFunc, this, server, penID, this.id );
		});
	}

	clearOnRefresh_{ arg bool;
		clearOnRefresh = bool;
		this.setProperty( \clearOnRefresh, bool );
	}

	drawFunc_ { arg func;
		if( drawFunc.isNil, {
			if( func.notNil, {
				penID	= server.nextNodeID;
				server.sendBundle( nil,
					[ '/local', penID, '[', '/new', "de.sciss.swingosc.Pen", '[', '/ref', this.id, ']', relativeOrigin.not, ']' ],
					[ '/method', this.id, \setPen, '[', '/ref', penID, ']' ]
				);
			}, {
				^this;
			});
		}, {
			if( func.isNil, {
				server.sendBundle( nil,
					[ '/method', this.id, \setPen, '[', '/ref', \null, ']' ],
					[ '/method', penID, \dispose ],
					[ '/free', penID ]
				);
				penID = nil;
				drawFunc = nil;
				pendingDraw = false;
				^this;
			});
		});
		drawFunc = func;
		if( this.prAllVisible, {
			pendingDraw = false;
			JPen.protRefresh( drawFunc, this, server, penID, this.id );
		}, {
			pendingDraw = true;
		});
	}
	
	focusVisible { ^this.getProperty( \focusVisible, true )}
	focusVisible_ { arg visible; this.setProperty( \focusVisible, visible )}

	// ----------------- private instance methods -----------------

//	draw {
//		this.refresh;
//	}

	prFocusChange {
		// the user may wish to paint differently according to the focus
		if( refreshOnFocus, { this.protDraw });
	}
	
	prVisibilityChange {
		if( pendingDraw, { this.protDraw });
	}

	prBoundsUpdated {
		this.protDraw;
	}

	prClose { arg preMsg, postMsg;
		this.drawFunc_( nil );
		^super.prClose( preMsg, postMsg );
	}

	protDraw {
		if( drawFunc.notNil and: { this.prAllVisible }, {
//			// cmpID == nil --> don't repaint, because this
//			// will be done already by JSCWindow, and hence
//			// would slow down refresh unnecessarily (???)
//			JPen.protRefresh( drawFunc, this, server, penID, nil );
			JPen.protRefresh( drawFunc, this, server, penID, this.id );
		});
	}
}

JSCUserView : JSCAbstractUserView {
	var lastMouseX, lastMouseY;

	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		^this.new( parent, bounds ).refreshOnFocus_( false ).drawFunc_({ arg view;
			var b = view.bounds, min = min( b.width, b.height ), max = max( b.width, b.height ),
			    num = (max / min).asInteger;
			JPen.addRect( b );
			JPen.clip;
			JPen.translate( b.left, b.top );
			JPen.scale( min, min );
			num.do({ 	arg i;
				var rel = i / num;
				JPen.fillColor = Color.hsv( rel, 0.4, 0.6 );
				JPen.addWedge( (0.5 + i) @ 0.5, 0.4, rel * pi + 0.2, 1.5pi );
				JPen.fill;
			});
		});
	}

	// ----------------- public instance methods -----------------

	mousePosition {
		var b = this.bounds;
		^((lastMouseX - b.left) @ (lastMouseY - b.top));
	}

	relativeOrigin_ { arg bool;
		relativeOrigin = bool;
		this.setProperty( \relativeOrigin, bool );
	}

	// ----------------- private instance methods -----------------

	mouseDown { arg x, y ... rest;
		lastMouseX	= x;
		lastMouseY	= y;
		^super.mouseDown( x, y, *rest );
	}
	
	mouseUp { arg x, y ... rest;
		lastMouseX	= x;
		lastMouseY	= y;
		^super.mouseUp( x, y, *rest );
	}
	
	mouseMove { arg x, y ... rest;
		lastMouseX	= x;
		lastMouseY	= y;
		^super.mouseMove( x, y, *rest );
	}
	
	mouseOver { arg x, y ... rest;
		lastMouseX	= x;
		lastMouseY	= y;
		^super.mouseOver( x, y, *rest );
	}

	prSCViewNew {
		relativeOrigin	= false;
		jinsets			= Insets( 3, 3, 3, 3 );
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.UserView", ']' ]
		]);
	}

	prSendProperty { arg key, value;

		key	= key.asSymbol;

		// fix keys
		case { key === \relativeOrigin }
		{
			if( penID.notNil, { server.sendMsg( '/set', penID, \absCoords, value.not )});
			^nil;
		};
		^super.prSendProperty( key, value );
	}
}

JSCTextView : JSCView {

	var <stringColor, <font, <editable = true;
	var <autohidesScrollers = false, <hasHorizontalScroller = false, <hasVerticalScroller = false;
	var <usesTabToFocusNextView = true, <enterInterpretsSelection = true;
//	var <textBounds;

	var txResp;
	
	var <string = "";
	var selStart = 0, selStop = 0;
	
	var cnID;

//	mouseDown { arg clickPos;
////		this.focus(true);
//		mouseDownAction.value( this, clickPos );	
//	}	
	
//	string {
//		^this.getProperty( \string );
//	}

	// ----------------- public instance methods -----------------

	string_ { arg str;
		^this.setString( str, -1 );
	}
		
	selectedString {
		^string.copyRange( selStart, selStop - 1 );  // stupid inclusive ending
	}
	
	selectedString_{ arg str;
		this.setProperty( \selectedString, str );
		// XXX
	}
	
	selectionStart { ^selStart }
	selectionSize { ^(selStop - selStart) }
	
	stringColor_ { arg color;
		stringColor = color;
		this.setStringColor( color, -1, 0 );
	}
	
	setStringColor { arg color, rangeStart = -1, rangeSize = 0;
		server.listSendMsg([ '/method', this.id, \setForeground, rangeStart, rangeSize ] ++ color.asSwingArg );
	}
	
	font_ { arg afont;
		font = afont;
		this.setFont( font, -1, 0 );
	}
	
	setFont { arg font, rangestart = -1, rangesize = 0;
		server.listSendMsg([ '/method', this.id, \setFont, rangestart, rangesize ] ++ font.asSwingArg );
	}
	
	setString { arg string, rangestart = 0, rangesize = 0;
		var bndl, off, len, bndlSize;
	
//		string		= string.asString;
		
		// server.options.oscBufSize - sizeof([ '/method', 1234, \setString, 0, 1, "" ])
		if( string.size <= (server.options.oscBufSize - 44), {
			server.sendMsg( '/method', this.id, \setString, rangestart, rangesize, string );
		}, {
			bndl	= Array( 3 );
			off	= 0;
			// [ #bundle, [ '/method', 1234, \beginDataUpdate ],
			//            [ '/method', 1234, \addData, "GA" ],
			//            [ '/method', 1234, \endDataUpdate, 0, 1 ]
			bndlSize = 136;
			bndl.add([ '/method', this.id, \beginDataUpdate ]);
			while({ off < string.size }, {
				len = min( string.size - off, server.options.oscBufSize - bndlSize );
				bndl.add([ '/method', this.id, \addData, string.copyRange( off, off + len - 1 )]);				off = off + len;
				if( off < string.size, {
					server.listSendBundle( nil, bndl );
					bndl = Array( 2 );
					bndlSize = 100; // wie oben, jedoch ohne \beginDataUpdate
				});
			});
			bndl.add([ '/method', this.id, \endDataUpdate, rangestart, rangesize ]);
			server.listSendBundle( nil, bndl );
		});
	}
	
	editable_ { arg bool;
		editable = bool;
		server.sendMsg( '/set', this.id, \editable, bool );
	}
	
	usesTabToFocusNextView_ { arg bool;
		usesTabToFocusNextView = bool;
		this.setProperty( \usesTabToFocusNextView, bool );
	}
	
	enterInterpretsSelection_ { arg bool;
		enterInterpretsSelection = bool;
		this.setProperty( \enterExecutesSelection, bool );
	}
	
	autohidesScrollers_ { arg bool;
		autohidesScrollers = bool;
		this.prUpdateScrollers;
	}
	
	hasHorizontalScroller_{ arg bool;
		hasHorizontalScroller = bool;
		this.prUpdateScrollers;
	}
	
	hasVerticalScroller_{ arg bool;
		hasVerticalScroller = bool;
		this.prUpdateScrollers;
	}
	
// what's the point about this method??
//	textBounds_{ arg rect;
//		textBounds = rect;
//		this.setProperty(\textBounds, rect);
//	}

	caretColor { ^this.getProperty( \caretColor )}
	caretColor_ { arg color; this.setProperty( \caretColor, color )}

	openURL { arg url;
		server.sendMsg( '/method', this.id, \setPage, '[', '/new', "java.net.URL", url, ']' );
		// XXX update client send string rep.
	}

	defaultKeyDownAction { arg key, modifiers, unicode;
		// check for 'ctrl+enter' = interprete
		if( (unicode == 0x0D) and: { ((modifiers & 0x40000) != 0) && enterInterpretsSelection }, {
			if( selStop > selStart, {	// text is selected
				this.selectedString.interpretPrint;
			}, {
				this.prCurrentLine.interpretPrint;
			});
			^this;
		});
		^nil;
	}

	// ----------------- private instance methods -----------------
	
	prContainerID { ^cnID }

	prSCViewNew {
		cnID = "cn" ++this.id;
//		properties.put( \value, 0 );
		
		txResp = OSCpathResponder( server.addr, [ '/doc', this.id ], { arg time, resp, msg;
			var state, str;
			
			state = msg[2];
	
			case
			{ state === \insert }
			{
//				("insert at "++msg[3]++" len "++msg[4]++" text='"++msg[5]++"'").postln;
				str = msg[5].asString;
if( msg[4] != str.size, { ("Yukk. len is "++msg[4]++"; but string got "++str.size).postln });
				string = string.insert( msg[3], str );
				if( action.notNil, {{ action.value( this, state, msg[3], msg[4], str )}.defer });
			}
			{ state === \remove }
			{
//				("remove from "++msg[3]++" len "++msg[4]).postln;
				string = string.keep( msg[3] ) ++ string.drop( msg[3] + msg[4] );
				if( action.notNil, {{ action.value( this, state, msg[3], msg[4] )}.defer });
			}
			{ state === \caret }
			{
//				("caret now between "++msg[3]++" and "++msg[4]).postln;
				if( msg[3] < msg[4], {
					selStart	= msg[3];
					selStop	= msg[4];
				}, {
					selStart	= msg[4];
					selStop	= msg[3];
				});
				if( action.notNil, {{ action.value( this, state, msg[3], msg[4] )}.defer });
			};
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.TextView", ']',
				"cn" ++ this.id,				 	// bars : v=never, h=never
				'[', '/new', "javax.swing.JScrollPane", '[', '/ref', this.id, ']', 21, 31, ']',
				"tx" ++ this.id,
				'[', '/new', "de.sciss.swingosc.DocumentResponder", this.id, ']'
			]
		]);
	}

	prUpdateScrollers {
		server.sendMsg( '/set', "cn" ++ this.id,
			\horizontalScrollBarPolicy, hasHorizontalScroller.if( autohidesScrollers.if( 30, 32 ), 31 ),
			\verticalScrollBarPolicy, hasVerticalScroller.if( autohidesScrollers.if( 20, 22 ), 21 ));
	}

	prClose { arg preMsg, postMsg;
		txResp.remove;
		^super.prClose( preMsg ++ [[ '/method', "tx" ++ this.id, \remove ]], postMsg );
	}

	prCurrentLine {
		var startIdx, stopIdx;
		
		startIdx	= string.findBackwards( "\n", false, selStart - 1 ) ? 0;
		stopIdx	= string.find( "\n", false, selStart ) ?? { string.size };
		^string.copyRange( startIdx, stopIdx - 1 );
	}
}

JSCAbstractMultiSliderView : JSCView { 

	var <>metaAction;
	var <size = 0;
		
	// ----------------- public instance methods -----------------

	step_ { arg stepSize; this.setPropertyWithAction( \step, stepSize )}
	
	step { ^this.getProperty( \step )}
	
	selectionSize { ^this.getProperty( \selectionSize )}

	selectionSize_ { arg aval; this.setProperty( \selectionSize, aval )}

	currentvalue { // returns value of selected index
		^this.getProperty( \y );
	}
	
	strokeColor_ { arg acolor; this.setProperty( \strokeColor, acolor )}

	currentvalue_ { arg iny; this.setProperty( \y, iny )}
	
	drawLines { arg abool; this.setProperty( \drawLines, abool )}

	drawLines_ { arg abool; this.drawLines( abool )}
	
	drawRects_ { arg abool; this.setProperty( \drawRects, abool )}

	doMetaAction { // performed on ctrl click
		metaAction.value( this );
	} 

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \value, \strokeColor, \x, \y, \drawLines, \drawRects, \selectionSize, \step ]; // JJJ not thumbSize, thumbWidth, not absoluteX
	}
	
	defaultCanReceiveDrag {	^true }
			
	prNeedsTransferHandler { ^true }
}

JSCMultiSliderView : JSCAbstractMultiSliderView { 

	var acResp;	// OSCpathResponder for action listening
	var vlResp;	// OSCpathResponder for value update listening
	var clpse;

	var <gap;
	var <editable = true;
	var <elasticMode = 0;
		
	// ----------------- public instance methods -----------------

	elasticMode_{ arg mode;
		elasticMode = mode;
		this.setProperty( \elasticResizeMode, mode );
	}

	value { // returns array
		^this.getProperty( \value, Array.newClear( this.size ));
	}
	
	value_ { arg val;
		size = val.size;
		this.setProperty( \value, val.copy );
	}

	valueAction_ { arg val;
		size = val.size;	
		this.setPropertyWithAction( \value, val.copy );
	}
	
	reference { // returns array
		^this.getProperty( \referenceValues, Array.newClear( this.size ));
	}
	
	reference_ { arg val;
		// this.size = val.size;
		this.setProperty( \referenceValues, val );
	}
	
	index { // returns selected index
		^this.getProperty( \x );
	}
	
	index_ { arg inx;
		this.setProperty( \x, inx );
	}
	
	fillColor_ { arg acolor; this.setProperty( \fillColor, acolor )}

	colors_ { arg strokec, fillc;
		this.strokeColor_( strokec );
		this.fillColor_( fillc );
	}
	
	isFilled_ { arg abool;
		this.setProperty( \isFilled, abool );
	}
	
	xOffset_ { arg aval;
		this.setProperty( \xOffset, aval );
	}
	
	gap_ { arg inx;
		gap = inx;
		this.setProperty( \xOffset, inx );
	}
	
	startIndex_ { arg val; this.setProperty( \startIndex, val )}
	
	showIndex_ { arg abool; this.setProperty( \showIndex, abool )}
	
	// = thumb width
	indexThumbSize_ { arg val; this.setProperty( \thumbWidth, val )}

	// = thumb height
	valueThumbSize_ { arg val; this.setProperty( \thumbHeight, val )}

	indexIsHorizontal_ { arg val; this.setProperty( \isHorizontal, val )}
	
	thumbSize_ { arg val;
		properties.put( \thumbWidth, val );
		properties.put( \thumbHeight, val );
		server.sendMsg( '/set', this.id, \thumbSize, val );
	}
	
	readOnly_ { arg val;
		editable = val.not;
		this.setProperty( \readOnly, val );
	}
	
	editable_ { arg val;
		editable = val;
		this.setProperty( \readOnly, editable.not );
	}
	
	defaultReceiveDrag {
		if( currentDrag[ 0 ].isSequenceableCollection, { 
			this.value_( currentDrag[ 0 ]);
			this.reference_( currentDrag[ 1 ]);
		}, {
			this.value_( currentDrag );
		});
	}
	
	defaultGetDrag {
		var setsize, vals, rvals, outval;
		rvals = this.reference;
		vals = this.value;
		if( this.selectionSize > 1, {
			vals = vals.copyRange( this.index, this.selectionSize + this.index );
		});
		if( rvals.isNil, { 
			^vals; 
		}, {
			if( this.selectionSize > 1, {
				rvals = rvals.copyRange( this.index, this.selectionSize + this.index );
			});
			outval = outval.add( vals );
			outval = outval.add( rvals );
		});
		^outval;
	}
		
	defaultKeyDownAction { arg key, modifiers, unicode;
		//modifiers.postln; 16rF702
		if (unicode == 16rF703, { this.index = this.index + 1; ^this });
		if (unicode == 16rF702, { this.index = this.index - 1; ^this });
		if (unicode == 16rF700, { this.gap = this.gap + 1; ^this });
		if (unicode == 16rF701, { this.gap = this.gap - 1; ^this });
		^nil		// bubble if it's an invalid key
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \elasticResizeMode, \fillColor, \thumbWidth, \thumbHeight, \xOffset, \showIndex, \startIndex, \referenceValues, \isFilled, \readOnly ];	// JJJ not \thumbSize, but \thumbHeight, added \readOnly
	}
		
	prClose { arg preMsg, postMsg;
		vlResp.remove;
		acResp.remove;
		clpse.cancel;
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		var initVal;
		initVal	= 0 ! 8;
		properties.put( \value, initVal );
		properties.put( \x, 0 );
		properties.put( \y, 0.0 );
		properties.put( \step, 0.0 );
		jinsets	= Insets( 3, 3, 3, 3 );
		clpse	= Collapse({ this.doAction });
		vlResp	= OSCpathResponder( server.addr, [ '/values', this.id ], { arg time, resp, msg;
			var dirtyIndex, dirtySize, vals, selectedIndex;

			vals			= properties[ \value ];
			dirtyIndex	= min( msg[ 2 ], vals.size );
			dirtySize		= min( msg[ 3 ], vals.size - dirtyIndex );
			
			dirtySize.do({ arg i;
				vals[ dirtyIndex + i ] = msg[ 4 + i ];
			});
			selectedIndex	= this.getProperty( \x, -1 );
//("selectedIndex = "++selectedIndex++"; vals = "++vals).inform;
			if( (selectedIndex >= dirtyIndex) and: { selectedIndex < (dirtyIndex + dirtySize) }, {
				properties.put( \y, vals[ selectedIndex ]);
			});
			if( dirtySize > 0, { clpse.instantaneous });
		}).add;
		acResp = OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var dirtyIndex, dirtySize, vals, selectedIndex;

			selectedIndex	= msg[ 4 ];
			properties.put( \x, selectedIndex );
			properties.put( \selectionSize, msg[ 6 ]);
			dirtyIndex	= msg[ 8 ];
			dirtySize		= msg[ 10 ];

			if( dirtySize == 0, {
				vals = properties[ \value ];
				properties.put( \y, vals[ selectedIndex ]);
				clpse.instantaneous;
			});
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.MultiSlider", ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionMessenger",  // ActionResponder
					this.id, '[', '/array', \selectedIndex, \selectionSize, \dirtyIndex, \dirtySize, ']',
					\sendValuesAndClear, '[', '/array', this.id, ']', ']' ],
			[ '/set', this.id, \values ] ++ initVal.asSwingArg
		]);
	}

	prSendProperty { arg key, value;
		key	= key.asSymbol;

		// fix keys
		switch( key,
		\value, {
			key = \values;
			this.prFixValues;
		},
		\x, {
			key = \selectedIndex;
		},
		\isFilled, {
			key = \filled;
		},
		\isHorizontal, {
			key 		= \orientation;
			value	= if( value, 0, 1 );
		},
		\step, {
			key = \stepSize;
			this.prFixValues;
		});
		^super.prSendProperty( key, value );
	}
	
	prFixValues {
		var val, step;
		
		val	= properties[ \value ];
		step	= this.step;
		if( step > 0, {
			val.size.do({ arg i; val[ i ] = val[ i ].round( step ).clip( 0.0, 1.0 )});
		}, {
			val.size.do({ arg i; val[ i ] = val[ i ].clip( 0.0, 1.0 )});
		});
	}
}

JSCEnvelopeView : JSCAbstractMultiSliderView {
	var allConnections, selection;
	var items;
	var connectionsUsed = false;
	var idx = 0;	// the one that corresponds to select, x_ and y_
	
	var acResp;
	var vlResp;
	var clpse;

// rather useless behaviour in SCEnvelopeView (using shift+click you can ignore it),
// so keep it out for now
//	var <fixedSelection = false;
	
	// ----------------- public class methods -----------------

	*paletteExample { arg parent, bounds;
		^this.new( parent, bounds ).value_([ (0..4)/4, sqrt( (0..4)/4 )])
			.thumbSize_( 4 ).drawLines_( true ).selectionColor_( Color.red );
	}

	// ----------------- public instance methods -----------------

	value_ { arg val;
		var oldSize, xvals, yvals, curves, valClip;
		
		oldSize	= size;
		xvals	= val[ 0 ];
		yvals	= val[ 1 ];
		curves	= val[ 2 ];
		if( xvals.size != yvals.size, {
			Error( "JSCEnvelopeView got mismatched times/levels arrays" ).throw;
		});
		size 	= xvals.size;
		case
		{ oldSize < size }
		{
			if( allConnections.notNil, {
				allConnections = allConnections.growClear( size );
			});
			if( items.notNil, {
				items = items.growClear( size );
			});
			selection = selection.growClear( size ).collect({ arg sel; if( sel.isNil, false, sel )});
		}
		{ oldSize > size }
		{
			if( allConnections.notNil, {
				allConnections = allConnections.copyFromStart( size - 1 );
			});
			if( items.notNil, {
				items = items.copyFromStart( size - 1 );
			});
			selection = selection.copyFromStart( size - 1 );
		};

		if( curves.isNil, {
			valClip = [ xvals.collect({ arg x; x.clip( 0.0, 1.0 )}), yvals.collect({ arg y; y.clip( 0.0, 1.0 )})];
		}, {
			valClip = [ xvals.collect({ arg x; x.clip( 0.0, 1.0 )}), yvals.collect({ arg y; y.clip( 0.0, 1.0 )}), curves ];
		});
		this.setProperty( \value, valClip );
	}
	
	setString { arg index, astring;
		if( items.isNil, {
			items = Array.newClear( size );
		});
		if( index < 0, {
			items.fill( astring );
		}, { if( index < size, {
			items[ index ] = astring;
		})});
		// items = items.add( astring );
//		this.setProperty( \string, [ index, astring ]);
		server.listSendMsg([ '/method', this.id, \setLabel, index ] ++ astring.asSwingArg );
	}

	strings_ { arg astrings;
		astrings.do({ arg str,i;
//			this.string_( i, str );
			this.setString( i, str );
		});
	}
	
	strings {
		^items.copy;	// nil.copy allowed
	}
	
//	items_ { arg items; ^this.strings_( items )}
	
	value {
//		var ax, ay, axy;
//		ax = Array.newClear( this.size );
//		ay = Array.newClear( this.size );
//		axy = Array.with( ax, ay );
//		^this.getProperty( \value, axy );
		^properties[ \value ].deepCopy;
	}
	
	selection {
		^selection.copy;
	}
	
	connections {
		var result;
		if( allConnections.isNil, { ^nil });
		
		result = Array( allConnections.size );
		allConnections.do({ arg cons; result.add( cons.copy )});
		^result;
	}
	
	setThumbHeight { arg index, height;
//		this.setProperty( \thumbHeight, [ index, height ]);
		server.sendMsg( '/method', this.id, \setThumbHeight, index, height );
	}
	
	thumbHeight_ { arg height; this.setThumbHeight( -1, height )}
	
	setThumbWidth { arg index, width;
//		this.setProperty( \thumbWidth, [ index, width ]);
		server.sendMsg( '/method', this.id, \setThumbWidth, index, width );
	}

	thumbWidth_ { arg width; this.setThumbWidth( -1, width )}

	setThumbSize { arg index, size;
//		this.setProperty(\thumbSize, [index, size]);
		server.sendMsg( '/method', this.id, \setThumbSize, index, size );
	}
	
	thumbSize_ { arg size; this.setThumbSize( -1, size )}

	setFillColor { arg index, color;
//		this.setProperty(\fillColor, [index, color]);
		server.listSendMsg([ '/method', this.id, \setFillColor, index ] ++ color.asSwingArg );
	}

	fillColor_ { arg color; this.setFillColor( -1, color )}

	colors_ { arg strokec, fillc;
		this.strokeColor_( strokec );
		this.fillColor_( fillc );
	}
	
	setCurve { arg index, curve = \lin;
		var shape, value, curves;
		value  = properties[ \value ];
		curves = value[ 2 ];
		if( index == -1, {
			if( curves.notNil, {
				curves.fill( curve );
			}, {
//				("AAA " ++ value).postln;
				properties[ \value ] = value ++ [(curve ! value[1].size)];
//				("AAA " ++ properties[ \value ]).postln;
			});
		}, { if( index < size, {
			if( curves.notNil, {
				curves[ index ] = curve;
			}, {
//				("BBB " ++ value).postln;
				properties[ \value ] = value ++ [(\lin ! value[1].size).put( index, curve )];
//				("BBB " ++ properties[ \value ]).postln;
			});
		})});
		if( curve.isFloat, {
			shape = 5;
		}, {
			shape = Env.shapeNames[ curve ];
			curve = 0.0;
		});
		server.sendMsg( '/method', this.id, \setShape, index, shape, curve );
	}
	
	curve_ { arg curve = \lin; this.setCurve( -1, curve )}

	lockBounds_ { arg val; this.setProperty( \lockBounds, val )}
	
	horizontalEditMode_ { arg val; this.setProperty( \horizontalEditMode, val )}
	
	connect { arg from, aconnections;
		var bndl, target, targetCons, fromCons;

		if( (from < 0) || (from >= size), { ^this });

		bndl			= Array( aconnections.size + 1 ); // max. number of messages needed
		fromCons		= Array( aconnections.size );

		if( connectionsUsed.not, {
			bndl.add([ '/set', this.id, \connectionsUsed, true ]);
			connectionsUsed	= true;
			allConnections	= Array.newClear( size );
		});

		aconnections.do({ arg target;
			target = target.asInteger;
			if( (target >= 0) && (target < size) && (target != from), {
				fromCons.add( target );
				targetCons = allConnections[ target ];
				if( targetCons.isNil or: { targetCons.includes( from ).not }, {
					targetCons = targetCons ++ [ from ];
					allConnections[ target ] = targetCons;
					// don't draw connections twice, so simply set only connections on the server whose target idx is greater than from idx
					bndl.add([ '/method', this.id, \setConnections, target ] ++ targetCons.reject({ arg idx; idx < target }).asSwingArg );
				});
			});
		});
		allConnections[ from ] = fromCons;
		bndl.add([ '/method', this.id, \setConnections, target ] ++ fromCons.reject({ arg idx; idx < from }).asSwingArg );
		server.listSendBundle( nil, bndl );
	}

	select { arg index; // this means no refresh;
		var vals;
//		this.setProperty(\setIndex, index);
		idx = index;
		if( (idx >= 0) && (idx < size), {
			vals = properties[ \value ];
			properties.put( \x, vals[ 0 ][ index ]);
			properties.put( \y, vals[ 1 ][ index ]);
		});
		server.sendMsg( '/set', this.id, \index, index );
	}
	
	selectIndex { arg index; // this means that the view will be refreshed
//		this.setProperty( \selectedIndex, index );
		properties.put( \selectedIndex, index );
		if( (idx >= 0) && (idx < size), {
			selection[ index ] = true;
		});
		server.sendMsg( '/method', this.id, \setSelected, index, true );
	}
	
	deselectIndex { arg index; // this means that the view will be refreshed
//		properties.put( \selectedIndex, index );
		if( (idx >= 0) && (idx < size), {
			selection[ index ] = false;
		});
		server.sendMsg( '/method', this.id, \setSelected, index, false );
	}
	
	x { ^this.getProperty( \x )}  // returns selected x
	y { ^this.getProperty( \y )}

	x_ { arg ax;
		ax = ax.round( this.step ).clip( 0.0, 1.0 );
		if( idx == -1, {
			properties[ \value ][ 0 ].fill( ax );
		}, { if( idx < size, {
			properties[ \value ][ 0 ][ idx ] = ax;
		})});
		this.setProperty( \x, ax );
	}

	y_ { arg ay;
		ay = ay.round( this.step ).clip( 0.0, 1.0 );
		if( idx == -1, {
			properties[ \value ][ 1 ].fill( ay );
		}, { if( idx < size, {
			properties[ \value ][ 1 ][ idx ] = ay;
		})});
		this.setProperty( \y, ay )
	}

	index { ^this.getProperty( \selectedIndex )}

//	lastIndex { ^this.getProperty( \lastIndex )}

	setEditable { arg index, boolean;
//		this.setProperty(\editable, [index,boolean]);
		server.sendMsg( '/method', this.id, \setReadOnly, index, boolean.not );
	}

	editable_{ arg boolean; this.setEditable( -1, boolean )}	
	selectionColor_ { arg acolor; this.setProperty( \selectionColor, acolor )}
	
// currently broken in cocoa
/*
	addValue { arg xval, yval;
		var arr, arrx, arry, aindx;
		// XXX could use custom server method!!
		aindx = this.lastIndex;
//		aindx.postln;
		if( xval.isNil && yval.isNil, {
			arr = this.value;
			arrx = arr @ 0;
			arry = arr @ 1;
			xval = arrx[ aindx ] + 0.05;
			yval = arry[ aindx ];
		});
		if( aindx < (arrx.size - 1), {
			arrx = arrx.insert( aindx + 1, xval );
			arry = arry.insert( aindx + 1, yval );
		}, {
			arrx = arrx.add( xval );
			arry = arry.add( yval );
		});		
		this.value_([ arrx, arry ]);
	}
*/

// see comment for <fixedSelection	
//	fixedSelection_ { arg bool;
//		fixedSelection =  bool;
//		this.setProperty(\setFixedSelection, bool);
//	}

	font { ^this.getProperty( \font )}

	font_ { arg argFont;
		this.setProperty( \font, argFont );
	}
	
	clipThumbs { ^this.getProperty( \clipThumbs )}

	clipThumbs_ { arg bool; this.setProperty( \clipThumbs, bool )}

	defaultGetDrag { ^this.value }

// currently broken in cocoa
/*
	defaultReceiveDrag {
		if( currentDrag.isString, {
			this.addValue;
//			items = items.insert( this.lastIndex + 1, currentDrag );
//			this.strings_( items );
			this.setString( this.lastIndex + 1, currentDrag );
		}, {
			this.value_( currentDrag );
		});
	}
*/
	defaultReceiveDrag { }
	
	defaultKeyDownAction { arg key, modifiers, unicode;
		var oldIdx, selIdx;

// gap is not working with envelope view!
//		if (unicode == 16rF700, { this.gap = this.gap + 1; ^this });
//		if (unicode == 16rF701, { this.gap = this.gap - 1; ^this });

		if( (unicode >= 16rF700) && (unicode <= 16rF703), {  // cursor
			selIdx	= this.index;
			oldIdx	= idx;
			if( (selIdx >= 0) and: { selIdx - 1 < this.size }, {
				case
				{ unicode == 16rF703 }	// cursor right
				{
					if( (modifiers & 524288) == 0, {	// test for alt
						this.select( selIdx );
						this.x = this.x + max( this.step, 0.015625 );
						this.select( oldIdx );
					}, { if( (selIdx + 1) < this.size, {
						this.deselectIndex( selIdx );
						this.selectIndex( selIdx + 1 );
					})});
				}
				{ unicode == 16rF702 }	// cursor left
				{
					if( (modifiers & 524288) == 0, {	// test for alt
						this.select( selIdx );
						this.x = this.x - max( this.step, 0.015625 );
						this.select( oldIdx );
					}, { if( selIdx > 0, {
						this.deselectIndex( selIdx );
						this.selectIndex( selIdx - 1 );
					})});
				}
				{ unicode == 16rF700 }	// cursor up
				{
					this.select( selIdx );
					this.y = this.y + max( this.step, 0.015625 );
					this.select( oldIdx );
				}
				{ unicode == 16rF701 }	// cursor down
				{
					this.select( selIdx );
					this.y = this.y - max( this.step, 0.015625 );
					this.select( oldIdx );
				};
			});
			^this;
		});
		^nil;		// bubble if it's an invalid key
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[ \font, \selectedIndex, \clipThumbs, \lockBounds, \horizontalEditMode ];  // \lastIndex
	}

	prClose { arg preMsg, postMsg;
		vlResp.remove;
		acResp.remove;
		clpse	= Collapse({ this.doAction });
		^super.prClose( preMsg ++
			[[ '/method', "ac" ++ this.id, \remove ],
			 [ '/free', "ac" ++ this.id ]], postMsg );
	}

	prSCViewNew {
		var initVal;
//		initVal	= nil ! 8 ! 2;	// pretty stupid
		initVal	= [[],[]];
		properties.put( \value, initVal );
		properties.put( \index, -1 );
		properties.put( \x, 0.0 );
		properties.put( \y, 0.0 );
//		properties.put( \lastIndex, -1 );	// 0 in cocoa ...
		properties.put( \selectedIndex, -1 );
		properties.put( \step, 0.0 );
		properties.put( \clipThumbs, false );
		selection	= [];
		jinsets	= Insets( 3, 3, 3, 3 );
		clpse	= Collapse({ this.doAction });
//		items	= Array.new;
		vlResp	= OSCpathResponder( server.addr, [ '/values', this.id ], { arg time, resp, msg;
			var dirtySize, vals, xvals, yvals, action = false;

			vals			= properties[ \value ];
			xvals		= vals[ 0 ];
			yvals		= vals[ 1 ];
			dirtySize		= msg[ 2 ];
			msg.copyToEnd( 3 ).clump( 4 ).do({ arg entry; var idx, x, y, sel;
				#idx, x, y, sel = entry;
				if( idx < xvals.size, {
					xvals[ idx ]		= x;
					yvals[ idx ]		= y;
					selection[ idx ]	= sel != 0;
					action			= true;
				});
			});
			if( action, { clpse.instantaneous });
		}).add;
		acResp = OSCpathResponder( server.addr, [ '/action', this.id ], { arg time, resp, msg;
			var lastIndex, dirtySize;

			lastIndex	= msg[ 4 ];
			if( lastIndex >= 0, { properties.put( \selectedIndex, lastIndex )}); // \lastIndex
			dirtySize	= msg[ 5 ];

//			if( dirtySize == 0, {
//				vals = properties[ \value ];
//				properties.put( \y, vals[ selectedIndex ]);
//				clpse.instantaneous;
//			});
		}).add;
		^super.prSCViewNew([
			[ '/local', this.id, '[', '/new', "de.sciss.swingosc.EnvelopeView", false, ']',
				"ac" ++ this.id,
				'[', '/new', "de.sciss.swingosc.ActionMessenger",  // ActionResponder
					this.id, '[', '/array', \lastIndex, \dirtySize, ']',
					\sendDirtyValuesAndClear, '[', '/array', this.id, ']', ']' ],
		]);
	}

	prSendProperty { arg key, value;
		var ival, shapes, curvesSC, curves;
		
		key	= key.asSymbol;

		// fix keys
		switch( key,
		\value, {
			this.prFixValues;
			if( value.size < 3, {
				server.listSendMsg([ '/method', this.id, \setValues ] ++ value[ 0 ].asSwingArg ++ value[ 1 ].asSwingArg );
			}, {
				curvesSC = value[ 2 ];
				if( curvesSC.isArray, {
					shapes = Array( curvesSC.size );
					curves = Array( curvesSC.size );
					curvesSC.do({ arg curve;
						if( curve.isFloat, {
							shapes.add( 5 );
							curves.add( curve );
						}, {
							shapes.add( Env.shapeNames[ curve ]);
							curves.add( 0.0 );
						});
					});
				}, {
					if( curvesSC.isFloat, {
						shapes = 5;
						curves = curvesSC;
					}, {
						shapes = Env.shapeNames[ curvesSC ];
						curves = 0.0;
					});
				});
				server.listSendMsg([ '/method', this.id, \setValues ] ++ value[ 0 ].asSwingArg ++ value[ 1 ].asSwingArg ++ shapes.asSwingArg ++ curves.asSwingArg );
			});
			^this;
		},
		\step, {
			key = \stepSize;
			this.prFixValues;
		},
		\horizontalEditMode, {
			ival = [ \free, \clamp, \relay ].indexOf( value );
			if( ival.isNil, { Error( "Illegal edit mode '" ++ value ++ "'" ).throw });
			value = ival;
		});
		^super.prSendProperty( key, value );
	}

	prFixValues {
		var val, step;
		
		val	= properties[ \value ];
		step	= this.step;
		if( step > 0, {
			2.do({ arg j; var xyvals = val[ j ]; xyvals.size.do({ arg i; xyvals[ i ] = xyvals[ i ].round( step ).clip( 0.0, 1.0 )})});
		}, {
			2.do({ arg j; var xyvals = val[ j ]; xyvals.size.do({ arg i; xyvals[ i ] = xyvals[ i ].clip( 0.0, 1.0 )})});
		});
	}
}

JSCTextEditBase : JSCStaticTextBase {

	var <>keyString;
	var	<>typingColor, <>normalColor;

	// ----------------- public instance methods -----------------

	caretColor { ^this.getProperty( \caretColor )}
	caretColor_ { arg color; this.setProperty( \caretColor, color )}

	value { ^object }
	
	value_ { arg val;
		keyString = nil;
		this.stringColor = normalColor;
		object = val;
		this.string = object.asString;
	}
	
	valueAction_ { arg val;
		var prev;
		prev = object;
		this.value = val;
		if (object != prev, { this.doAction });
	}
	
	boxColor {
		^this.getProperty( \boxColor, Color.new );
	}
	
	boxColor_ { arg color;
		this.setProperty( \boxColor, color );
	}

	// ----------------- private instance methods -----------------

	properties {
		^super.properties ++ #[\boxColor]
	}

	init { arg argParent, argBounds, id;
		typingColor = Color.red;
		normalColor = Color.black;
		parent = argParent.asView; // actual view
// cocoa does parent.asView once more. too cryptic IMO ?
//		this.prInit( parent, argBounds.asRect, this.class.viewClass, parent.server, id );
		this.prInit( parent.asView, argBounds.asRect, this.class.viewClass, parent.server, id );
		argParent.add( this );//maybe window or viewadapter
	}

	prSendProperty { arg key, value;
		key	= key.asSymbol;

		// fix keys
		case { key === \boxColor }
		{
			key = \background;
			if( value == Color.clear, {
				value = nil;
			});
		};
		^super.prSendProperty( key, value );
	}

	prNeedsTransferHandler { ^true }
}